<?php
session_start();
require_once '../config/db.php';
require_once '../includes/functions.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ../auth/login.php');
    exit;
}

// Get invoice ID from URL parameter
$invoice_id = $_GET['id'] ?? null;

if (!$invoice_id) {
    echo "Invoice ID is required";
    exit;
}

try {
    // Get invoice details with all related data
    $stmt = $pdo->prepare("
        SELECT i.*, c.name as customer_name, c.phone as customer_phone, c.email as customer_email, c.address as customer_address,
               u.username as created_by_name, q.quote_number as quotation_reference
        FROM invoices i
        LEFT JOIN customers c ON i.customer_id = c.id
        LEFT JOIN users u ON i.created_by = u.id
        LEFT JOIN quotations q ON i.quotation_id = q.id
        WHERE i.id = ?
    ");
    $stmt->execute([$invoice_id]);
    $invoice = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$invoice) {
        echo "Invoice not found";
        exit;
    }

    // Get invoice items
    $stmt = $pdo->prepare("
        SELECT ii.*, p.name as product_name
        FROM invoice_items ii
        LEFT JOIN products p ON ii.product_id = p.id
        WHERE ii.invoice_id = ?
    ");
    $stmt->execute([$invoice_id]);
    $items = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Get payment information
    $stmt = $pdo->prepare("
        SELECT SUM(amount) as total_paid
        FROM payments 
        WHERE invoice_id = ?
    ");
    $stmt->execute([$invoice_id]);
    $paymentData = $stmt->fetch(PDO::FETCH_ASSOC);
    $totalPaid = floatval($paymentData['total_paid'] ?? 0);
    $totalAmount = floatval($invoice['total_amount']);
    $balance = $totalAmount - $totalPaid;

    // Get store settings
    $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'store_info'");
    $stmt->execute();
    $storeSettings = [];
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $storeSettings[$row['setting_key']] = $row['setting_value'];
    }

    // Store information
    $storeName = $storeSettings['store_name'] ?? 'Your Store Name';
    $storeAddress = $storeSettings['store_address'] ?? 'Store Address';
    $storePhone = $storeSettings['store_phone'] ?? 'Phone Number';
    $storeEmail = $storeSettings['store_email'] ?? 'Email';
    $storeWebsite = $storeSettings['website'] ?? '';

} catch (Exception $e) {
    echo "Error loading invoice: " . $e->getMessage();
    exit;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Invoice <?php echo htmlspecialchars($invoice['invoice_number']); ?></title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Arial', sans-serif;
            line-height: 1.6;
            color: #333;
            background: #fff;
        }

        .invoice-container {
            max-width: 800px;
            margin: 20px auto;
            padding: 20px;
            border: 1px solid #ddd;
            background: #fff;
            position: relative;
            z-index: 1;
        }

        /* Watermark */
        .watermark {
            position: fixed;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            font-size: 18px;
            font-weight: bold;
            color: rgba(0, 0, 0, 0.15);
            z-index: 9999;
            text-align: center;
            pointer-events: none;
            width: 100%;
            height: 100%;
        }

        .watermark-content {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            width: 100%;
            height: 100%;
            line-height: 1.2;
        }

        .watermark-logo {
            max-width: 120px;
            max-height: 60px;
            margin-bottom: 20px;
            opacity: 0.15;
        }

        .header {
            text-align: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #333;
        }

        .store-logo {
            max-width: 150px;
            max-height: 80px;
            margin-bottom: 10px;
        }

        .store-name {
            font-size: 24px;
            font-weight: bold;
            margin-bottom: 5px;
        }

        .store-info {
            font-size: 14px;
            color: #666;
            line-height: 1.4;
        }

        .invoice-title {
            font-size: 28px;
            font-weight: bold;
            margin: 10px 0;
            text-align: center;
        }

        .invoice-details {
            display: flex;
            justify-content: space-between;
            margin-bottom: 20px;
        }

        .invoice-info, .customer-info {
            flex: 1;
        }

        .section-title {
            font-size: 16px;
            font-weight: bold;
            margin-bottom: 10px;
            color: #333;
            border-bottom: 1px solid #ddd;
            padding-bottom: 5px;
        }

        .info-row {
            margin-bottom: 8px;
            font-size: 14px;
        }

        .info-label {
            font-weight: bold;
            color: #555;
        }

        .items-table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 20px;
        }

        .items-table th {
            background: #f8f9fa;
            padding: 8px 8px;
            text-align: left;
            border-bottom: 2px solid #333;
            font-weight: bold;
        }

        .items-table td {
            padding: 6px 8px;
            border-bottom: 1px solid #ddd;
        }

        .items-table tr:nth-child(even) {
            background: #f9f9f9;
        }

        .text-right {
            text-align: right;
        }

        .text-center {
            text-align: center;
        }

        .summary-section {
            margin-top: 20px;
        }

        .summary-row {
            display: flex;
            justify-content: space-between;
            margin-bottom: 8px;
            font-size: 14px;
        }

        .summary-label {
            font-weight: bold;
        }

        .summary-value {
            text-align: right;
        }

        .total-row {
            font-size: 16px;
            font-weight: bold;
            border-top: 2px solid #333;
            padding-top: 10px;
            margin-top: 10px;
        }

        .payment-section {
            margin-top: 15px;
            padding-top: 15px;
            border-top: 1px solid #ddd;
        }

        .balance-due {
            color: #dc3545;
            font-weight: bold;
        }

        .footer {
            margin-top: 25px;
            text-align: center;
            font-size: 12px;
            color: #666;
            border-top: 1px solid #ddd;
            padding-top: 15px;
        }

        .notes {
            margin-top: 20px;
            padding: 15px;
            background: #f8f9fa;
            border-left: 4px solid #007bff;
        }

        .notes h4 {
            margin-bottom: 10px;
            color: #333;
        }

        @media print {
            body {
                margin: 0;
                padding: 0;
            }

            .invoice-container {
                border: none;
                margin: 0;
                padding: 20px;
            }

            .no-print {
                display: none;
            }
        }

        .print-button {
            position: fixed;
            top: 20px;
            right: 20px;
            background: #007bff;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 5px;
            cursor: pointer;
            font-size: 14px;
        }

        .print-button:hover {
            background: #0056b3;
        }
    </style>
</head>
<body>
    <button class="print-button no-print" onclick="window.print()">Print Invoice</button>
    
    <!-- Watermark -->
    <div class="watermark">
        <div class="watermark-content">
            <?php 
            // Get logo for watermark
            $watermarkLogo = getStoreLogo('watermark-logo', 'max-width: 120px; max-height: 60px; margin-bottom: 20px; opacity: 0.15;');
            echo $watermarkLogo;
            ?>
            <div style="font-size: 24px; margin-bottom: 10px; font-weight: bold; color: rgba(0, 0, 0, 0.2);"><?php echo htmlspecialchars($storeName); ?></div>
            <div style="font-size: 16px; margin-bottom: 8px; color: rgba(0, 0, 0, 0.18);"><?php echo htmlspecialchars($storeAddress); ?></div>
            <div style="font-size: 14px; margin-bottom: 6px; color: rgba(0, 0, 0, 0.18);"><?php echo htmlspecialchars($storePhone); ?></div>
            <?php if ($storeEmail): ?>
            <div style="font-size: 12px; margin-bottom: 5px; color: rgba(0, 0, 0, 0.18);"><?php echo htmlspecialchars($storeEmail); ?></div>
            <?php endif; ?>
            <?php if ($storeWebsite): ?>
            <div style="font-size: 11px; color: rgba(0, 0, 0, 0.18);"><?php echo htmlspecialchars($storeWebsite); ?></div>
            <?php endif; ?>
        </div>
    </div>

    <div class="invoice-container">
        <!-- Header -->
        <div class="header">
            <?php echo getStoreLogo('store-logo', 'max-width: 150px; max-height: 80px; margin-bottom: 10px;'); ?>
            
            <div class="store-name"><?php echo htmlspecialchars($storeName); ?></div>
            <div class="store-info">
                <?php echo htmlspecialchars($storeAddress); ?><br>
                <?php echo htmlspecialchars($storePhone); ?> | <?php echo htmlspecialchars($storeEmail); ?>
                <?php if ($storeWebsite): ?>
                    <br><?php echo htmlspecialchars($storeWebsite); ?>
                <?php endif; ?>
            </div>
        </div>

        <!-- Invoice Title -->
        <div class="invoice-title">INVOICE</div>

        <!-- Invoice and Customer Details -->
        <div class="invoice-details">
            <div class="invoice-info">
                <div class="section-title">INVOICE DETAILS</div>
                <div class="info-row">
                    <span class="info-label">Invoice Number:</span>
                    <span><?php echo htmlspecialchars($invoice['invoice_number']); ?></span>
                </div>
                <div class="info-row">
                    <span class="info-label">Issue Date:</span>
                    <span><?php echo date('M d, Y', strtotime($invoice['issue_date'])); ?></span>
                </div>
                <div class="info-row">
                    <span class="info-label">Due Date:</span>
                    <span><?php echo date('M d, Y', strtotime($invoice['due_date'])); ?></span>
                </div>
                <div class="info-row">
                    <span class="info-label">Status:</span>
                    <span><?php echo ucfirst($invoice['status'] ?? 'pending'); ?></span>
                </div>
                <?php if ($invoice['quotation_reference']): ?>
                <div class="info-row">
                    <span class="info-label">Quotation Ref:</span>
                    <span><?php echo htmlspecialchars($invoice['quotation_reference']); ?></span>
                </div>
                <?php endif; ?>
            </div>

            <div class="customer-info">
                <div class="section-title">BILL TO</div>
                <div class="info-row">
                    <span class="info-label">Customer:</span>
                    <span><?php echo htmlspecialchars($invoice['customer_name']); ?></span>
                </div>
                <div class="info-row">
                    <span class="info-label">Phone:</span>
                    <span><?php echo htmlspecialchars($invoice['customer_phone']); ?></span>
                </div>
                <div class="info-row">
                    <span class="info-label">Email:</span>
                    <span><?php echo htmlspecialchars($invoice['customer_email']); ?></span>
                </div>
                <div class="info-row">
                    <span class="info-label">Address:</span>
                    <span><?php echo htmlspecialchars($invoice['customer_address']); ?></span>
                </div>
            </div>
        </div>

        <!-- Items Table -->
        <table class="items-table">
            <thead>
                <tr>
                    <th>Item</th>
                    <th class="text-center">Quantity</th>
                    <th class="text-right">Unit Price</th>
                    <th class="text-right">Total</th>
                </tr>
            </thead>
            <tbody>
                <?php 
                $subtotal = 0;
                foreach ($items as $item): 
                    $itemTotal = $item['quantity'] * $item['unit_price'];
                    $subtotal += $itemTotal;
                ?>
                <tr>
                    <td><?php echo htmlspecialchars($item['product_name']); ?></td>
                    <td class="text-center"><?php echo $item['quantity']; ?></td>
                    <td class="text-right">KSh <?php echo number_format($item['unit_price'], 2); ?></td>
                    <td class="text-right">KSh <?php echo number_format($itemTotal, 2); ?></td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>

        <!-- Summary Section -->
        <div class="summary-section">
            <div class="summary-row">
                <span class="summary-label">Subtotal:</span>
                <span class="summary-value">KSh <?php echo number_format($subtotal, 2); ?></span>
            </div>
            <div class="summary-row">
                <span class="summary-label">Tax (0%):</span>
                <span class="summary-value">KSh <?php echo number_format($invoice['tax_amount'] ?? 0, 2); ?></span>
            </div>
            <div class="summary-row total-row">
                <span class="summary-label">Total:</span>
                <span class="summary-value">KSh <?php echo number_format($totalAmount, 2); ?></span>
            </div>
        </div>

        <!-- Payment Information -->
        <div class="payment-section">
            <div class="section-title">PAYMENT INFORMATION</div>
            <div class="summary-row">
                <span class="summary-label">Amount Paid:</span>
                <span class="summary-value">KSh <?php echo number_format($totalPaid, 2); ?></span>
            </div>
            <div class="summary-row">
                <span class="summary-label balance-due">Balance Due:</span>
                <span class="summary-value balance-due">KSh <?php echo number_format($balance, 2); ?></span>
            </div>
        </div>

        <!-- Notes -->
        <?php if (!empty($invoice['notes'])): ?>
        <div class="notes">
            <h4>Notes:</h4>
            <p><?php echo nl2br(htmlspecialchars($invoice['notes'])); ?></p>
        </div>
        <?php endif; ?>

        <!-- Footer -->
        <div class="footer">
            <p>This is a computer generated document and does not require signature.</p>
            <p>Generated on: <?php echo date('M d, Y \a\t g:i A'); ?></p>
            <p>Generated by: <?php echo htmlspecialchars($invoice['created_by_name']); ?></p>
        </div>
    </div>

    <script>
        // Auto-print when page loads
        window.onload = function() {
            // Small delay to ensure page is fully loaded
            setTimeout(function() {
                window.print();
            }, 500);
        };

        // Listen for print events
        window.addEventListener('afterprint', function() {
            // Close the tab and return to invoice page
            window.close();
        });

        // Also close tab when user cancels print (using beforeprint event)
        window.addEventListener('beforeprint', function() {
            // This will trigger when print dialog opens
        });

        // Alternative method: close tab after a delay if print dialog is closed
        let printTimeout;
        window.addEventListener('focus', function() {
            // Clear any existing timeout
            clearTimeout(printTimeout);
            
            // Set a timeout to close the tab after print dialog closes
            printTimeout = setTimeout(function() {
                window.close();
            }, 1000);
        });
    </script>
</body>
</html> 